/*
 * Copyright 2013 Israel Martín Escalona <imartin@entel.upc.edu>
 * 
 * This file is part of Wipos.
 * 
 * Wipos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Wipos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Wipos.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#undef __KERNEL__
#define __KERNEL__

#undef MODULE
#define MODULE

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h> /* printk */
#include <linux/fs.h> /* file_operations */
#include <linux/proc_fs.h> /* proc fs */
#include <linux/errno.h>
#include <linux/types.h>
#include <asm/uaccess.h> /* copy_from/to_user */
#include <linux/cdev.h> /* to register cdev file operations */
#include <positioning/globals.h>
#include <linux/if_ether.h> /* ETH_ALEN macro */
#include "pos80211.h"
#include "pos_ioctls.h"

//#define RUN_IF_NO_READERS(_code_) if (positioning_data_stats.number_of_readers == 0) {  _code_ }

#define TWO_WAY_TOA 0
#define PASSIVE_TDOA 1

#define DEFAULT_BUFFER_SIZE 256


/* Module information */
MODULE_AUTHOR(AUTHOR);
MODULE_DESCRIPTION(DESCRIPTION);
MODULE_VERSION(VERSION);
MODULE_LICENSE(LICENSE);

/* Module parameters */
static int mode;
module_param(mode, int, 0);

/* External function definition */
extern void clear_buffer_two_way_toa(void);
extern void set_gathering_buffer_two_way_toa(unsigned long);
extern void enable_gathering_two_way_toa(void);
extern void disable_gathering_two_way_toa(void);
extern read_return_struct get_next_rtt_two_way_toa(TYPE_OF_READ_ITEM * buffer, unsigned long size);

extern void clear_buffer_common_passive_tdoa(void);
extern void set_gathering_buffer_common_passive_tdoa(unsigned long);
extern void enable_gathering_common_passive_tdoa(void);
extern void disable_gathering_common_passive_tdoa(void);
extern read_return_struct get_next_passive_tdoa_measurement(TYPE_OF_READ_ITEM * buffer, unsigned long size);
extern void setActiveNodeAddressByStr(char * str);

/* Global variables */
static struct positioning_data_statistics_struct 
{
    unsigned int number_of_readers;
    unsigned long number_of_rtts_read;
    unsigned long number_of_buffered_rtts;
} positioning_data_stats;

static struct pos80211_ops
{
    void (*clear_gathering_buffer)(void);
    void (*set_gathering_buffer)(unsigned long);
    void (*enable_gathering_data)(void);
    void (*disable_gathering_data)(void);
    read_return_struct (*get_next_measurement)(TYPE_OF_READ_ITEM * buffer, unsigned long size);
} positioning_data_functions;


static dev_t dev;
static struct cdev * cdev;
static int major;

//TODO: PROC reading
//static struct proc_dir_entry *proc_file;
//static int positioning_data_proc_read(char *buffer, char **buffer_location, off_t offset, int buffer_length, int *eof, void *data);

/* function prototype */
static int positioning_data_open(struct inode *inode, struct file *filp);
static int positioning_data_release(struct inode *inode, struct file *filp);
static ssize_t positioning_data_read(struct file *filp, char *buf, size_t count, loff_t *f_pos);
long positioning_data_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);

/* Structure that declares the usual file access functions */
static struct file_operations fops = 
{
    .owner = THIS_MODULE,
    .read = positioning_data_read,
    .open = positioning_data_open,
    .release = positioning_data_release,
    .unlocked_ioctl = positioning_data_ioctl
};

/* INIT and EXIT of the module */
static int __init positioning_data_init(void) 
{   
    int result = SUCCESS;
    
    
    // Allocating major and minor
    result = alloc_chrdev_region(&dev, 0, 1, DEVICE_NAME);
    
    if (result < 0) 
    {
	printk(KERN_WARNING "%s: Can't get major\n", DEVICE_NAME);
	return result;
    }
    major = MAJOR(dev);
    
    // Allocating the character device struct
    cdev = cdev_alloc();
    if (!cdev)
    {
	printk(KERN_WARNING "%s: Unable to allocate character device\n", DEVICE_NAME);
	return -ENOMEM;
    }
    cdev->owner = THIS_MODULE;
    cdev->ops = &fops;
    
    // Adding the character device to the device definition
    cdev_add(cdev, dev, 1);
    
    // Selecting the mode
    switch (mode)
    {
	case TWO_WAY_TOA:
#ifdef _DEBUG_POSITIONING_	    
	    printk(KERN_INFO "%s: Two-way TOA mode selected\n", DEVICE_NAME);
#endif
	    positioning_data_functions.clear_gathering_buffer = clear_buffer_two_way_toa;
	    positioning_data_functions.set_gathering_buffer = set_gathering_buffer_two_way_toa;
	    positioning_data_functions.enable_gathering_data = enable_gathering_two_way_toa;
	    positioning_data_functions.disable_gathering_data = disable_gathering_two_way_toa;
	    positioning_data_functions.get_next_measurement = get_next_rtt_two_way_toa;
	    break;
	    
	case PASSIVE_TDOA:
#ifdef _DEBUG_POSITIONING_	    
	    printk(KERN_INFO "%s: Passive TDOA mode selected\n", DEVICE_NAME);
#endif
	    positioning_data_functions.clear_gathering_buffer = clear_buffer_common_passive_tdoa;
	    positioning_data_functions.set_gathering_buffer = set_gathering_buffer_common_passive_tdoa;
	    positioning_data_functions.enable_gathering_data = enable_gathering_common_passive_tdoa;
	    positioning_data_functions.disable_gathering_data = disable_gathering_common_passive_tdoa;
	    positioning_data_functions.get_next_measurement = get_next_passive_tdoa_measurement;
	    break;
	    
	default:
#ifdef _DEBUG_POSITIONING_	    
	    printk(KERN_INFO "%s: Two-way TOA mode selected (by default)\n", DEVICE_NAME);
#endif
	    positioning_data_functions.clear_gathering_buffer = clear_buffer_two_way_toa;
	    positioning_data_functions.set_gathering_buffer = set_gathering_buffer_two_way_toa;
	    positioning_data_functions.enable_gathering_data = enable_gathering_two_way_toa;
	    positioning_data_functions.disable_gathering_data = disable_gathering_two_way_toa;
	    positioning_data_functions.get_next_measurement = get_next_rtt_two_way_toa;
	    break;	    
    }
    // Initializing the positioning data buffering
    positioning_data_functions.disable_gathering_data();
    positioning_data_functions.set_gathering_buffer(DEFAULT_BUFFER_SIZE);
    
    // Initially, there are no positioning readers active
    positioning_data_stats.number_of_readers = 0;   
    positioning_data_stats.number_of_rtts_read = 0;
    positioning_data_stats.number_of_buffered_rtts = 0;
    
    // Module up statement
    printk(KERN_INFO "%s: Module is up with major %d!\n", DEVICE_NAME, major);
    
    return SUCCESS;
}

static void __exit positioning_data_exit(void) 
{
    // Freeing the character device
    cdev_del(cdev);
    
    // Freeing device
    unregister_chrdev_region(dev, 1);
    
    positioning_data_functions.disable_gathering_data();
    
    // Module down statement
    printk(KERN_ALERT "%s: Module is Down!\n", DEVICE_NAME);
    
}

// INIT THE MODULE

module_init(positioning_data_init);
module_exit(positioning_data_exit);


// FUNCTION DEFINITION
static int positioning_data_open(struct inode *inode, struct file *filp)
{
    // Only one reader is allowed. More than one might impact on the data precision. 
    
    if (positioning_data_stats.number_of_readers > 0)
	return -EBUSY;
    
    positioning_data_stats.number_of_readers++;
    
    try_module_get(THIS_MODULE);   
    
    return SUCCESS;
}

static int positioning_data_release(struct inode *inode, struct file *filp)
{
    positioning_data_stats.number_of_readers--;
    
    // Unsetting the active-node address
    if (mode == PASSIVE_TDOA)
	setActiveNodeAddressByStr("00:00:00:00:00:00");
    
    module_put(THIS_MODULE);
    
    return SUCCESS;
}

static ssize_t positioning_data_read(struct file *filp, char *buf, size_t count, loff_t *f_pos)
{   
    const int __BUFFER_SIZE = 32;
    unsigned long data_copied = 0;
    unsigned long number_of_samples = 0;
    unsigned long samples_to_read = 0;
    read_return_struct samples;
    int i = 0;
    TYPE_OF_READ_ITEM kBuf[__BUFFER_SIZE];
    
    
    /* Checking for suitable target address */
    if (!access_ok(VERIFY_WRITE, (char __user *)buf, count*sizeof(char)))
    {
#ifdef _DEBUG_POSITIONING_
	printk("%s: Memory address not valid\n", DEVICE_NAME);
#endif	
	return -EFAULT;
    }
    
    // The number of data to read has to be integer
    if (count % SIZE_OF_READ_ITEM != 0) 
    {
#ifdef _DEBUG_POSITIONING_
	printk("%s: Number of bytes %d is not multiple of the amount of samples\n", DEVICE_NAME, (int)count);
#endif
	return -EINVAL;
    }
    
    // Enabling the data gathering
    positioning_data_functions.clear_gathering_buffer();
    positioning_data_functions.enable_gathering_data();
    
    number_of_samples = samples_to_read = ((unsigned long)count / (unsigned long)SIZE_OF_READ_ITEM);
    while (samples_to_read > 0)
    {
	samples = positioning_data_functions.get_next_measurement(kBuf, samples_to_read > __BUFFER_SIZE ? __BUFFER_SIZE : samples_to_read);
	samples_to_read -= samples.data_read;

#ifdef _DEBUG_POSITIONING_
	printk("%s: %ld samples read\n", DEVICE_NAME, samples.data_read);
#endif	
	data_copied = copy_to_user((void *)buf, (void *)kBuf, samples.data_read * SIZE_OF_READ_ITEM);
	
	if (data_copied != 0)
	{
#ifdef _DEBUG_POSITIONING_
	    printk("%s: Error copying the positioning data\n", DEVICE_NAME);
#endif
	    positioning_data_functions.disable_gathering_data();
	    return ((number_of_samples - samples_to_read) * SIZE_OF_READ_ITEM - data_copied);
	}
	
	buf += samples.data_read * SIZE_OF_READ_ITEM;
	
	if (samples.error)
	{
#ifdef _DEBUG_POSITIONING_
	    if (samples.error != -ERESTARTSYS)
		printk("%s: Error %d reading the positioning data\n", DEVICE_NAME, samples.error);
#endif
	    positioning_data_functions.disable_gathering_data();
	    return ((number_of_samples - samples_to_read) * SIZE_OF_READ_ITEM) ;
	}
	
#ifdef _DEBUG_POSITIONING_
	printk("NEW BUFFERED:");
	for (i=0; i < samples.data_read; i++)
	    printk("%d %llu", kBuf[i].source, kBuf[i].item);
	printk("\n");
#endif
    }

    positioning_data_functions.disable_gathering_data();

    return (count);
}

long positioning_data_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    char addrk[3*ETH_ALEN];
    
    switch(cmd)
    {
	case POS80211_IOC_SET_ACTIVE_NODE:
	    // It only applies to passive TDOA
	    if (mode != PASSIVE_TDOA)
		return -ENOTTY;
	    
	    // Checking for suitable target address
	    if (!access_ok(VERIFY_READ, (char __user *)arg, 3*ETH_ALEN*sizeof(char)))
		return -EFAULT;
	    
	    // Getting the address string from user space
	    if (copy_from_user(addrk, (char __user *)arg, 3*ETH_ALEN*sizeof(char)))
		return -EFAULT;
	    
	    // Setting the active node according to the new address supplied by the user
	    setActiveNodeAddressByStr(addrk);
	    
	    break;
	case POS80211_IOC_RESET:
	    // TODO: Resetting the device
	    break;
	    
	default:
	    return -ENOTTY;
    }
    
    return SUCCESS;
}

// TODO: Change proc
// static int positioning_data_proc_read(char *buffer, char **buffer_location, off_t offset, int buffer_length, int *eof, void *data)
// {
//     int ret = 0;
//     
//     if (offset == 0) 
//     {
// 	/* filling the buffer and returning the buffer size */
// 	ret = sprintf(buffer, "Number of readers: %d\nNumber of RTT read: %lu\nNumber of buffered RTTs: %lu\n", positioning_data_stats.number_of_readers, positioning_data_stats.number_of_rtts_read, positioning_data_stats.number_of_buffered_rtts);
//     }
//     
//     return ret;
// }
