/*
 * Copyright 2013 Israel Martín Escalona <imartin@entel.upc.edu>
 * 
 * This file is part of Wipos.
 * 
 * Wipos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Wipos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Wipos.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <linux/slab.h>
#include <linux/limits.h>
#include "vector.h"

container_dato vGetNext_NoLock(Vector * v);
void vClear_NoLock(Vector * v);
void vReset_NoLock(Vector * v);


voidp allocateMemory(unsigned long size, char * callingFunction)
{
    voidp aM = (voidp) kmalloc(size, GFP_ATOMIC);
    
    if (!aM) {
        printk(KERN_ALERT "%s: kmalloc failed\n", callingFunction);
    }
    
    return aM;
}

Vector * vCreate(void)
{
    Vector * v = NULL;

    v = (Vector *) allocateMemory(sizeof(Vector), "vCreate");
    if (! v) return v;

    v->data = allocateMemory(sizeof(voidp) * VECTOR_DEFAULT_SIZE, "vCreate");
    if (!v->data) return NULL;

    spin_lock_init(&(v->properties.lock));
       
    vReset_NoLock(v);
    v->properties.size = 0;
    vSetIsPointer(v, false);
    vSetName(v, "No name");

    return v;
}

void vReset(Vector * v)
{
    vLock(v);
    vReset_NoLock(v);
    vUnlock(v);
}

void vReset_NoLock(Vector * v)
{
    v->properties.rPos = 0;
    v->properties.wPos = 0;
    v->properties.actualSize = 0;
}

void vSetName(Vector * v, char * name)
{
    
    int i = 0;
    
    vLock(v);
    
    while (i < MAX_CONTAINER_NAME_SIZE && *name != '\0') {
        v->properties.name[i] = *name;
        name++;
    }
    
    if (i == MAX_CONTAINER_NAME_SIZE && *name != '\0') {
        printk(KERN_ALERT "vSetName: name larger than %d characters\n", MAX_CONTAINER_NAME_SIZE-1);
        v->properties.name[MAX_CONTAINER_NAME_SIZE-1]='\0';
    }
    
    vUnlock(v);
    
    return;

}

// Queue must be empty!
void vDelete(Vector * v)
{
    vClear(v);
    
    kfree((voidp) v->data);
    kfree((voidp) v);
    v->data = NULL;
    
    return;
}

void vClear(Vector * v)
{
    vLock(v);
    vClear_NoLock(v);
    vUnlock(v);
    
    return;
}

inline void vClear_NoLock(Vector * v)
{
    container_dato dato;
    
    if (vIsPointer(v)) {
        while (vIsNotEmpty(v)) {
            dato = vGetNext_NoLock(v);
            kfree(dato.item);
        }
    }

    vReset_NoLock(v);
    
    return;
}


void vAddData(Vector * v, voidp item)
{
    if (v->properties.actualSize < v->properties.size) {
        vLock(v);
        v->properties.actualSize++;
        v->data[v->properties.wPos++] = (voidp) item;
        v->properties.wPos %= v->properties.size;
        vUnlock(v);
    }

    return;
}

inline container_dato vGetNext(Vector * v)
{
    container_dato dato;
  
    vLock(v);    
    dato = vGetNext_NoLock(v);
    vUnlock(v);

    return dato;
}

container_dato vGetNext_NoLock(Vector * v)
{
    container_dato dato;

    if (vIsNotEmpty(v)) {
        v->properties.actualSize--;
        dato.item = v->data[v->properties.rPos++];
        dato.error = 0;
        v->properties.rPos %= v->properties.size;
    } 
    else {
        dato.item = NULL;
        dato.error = -ERR_CONTAINER_IS_EMPTY;
    }

    return dato;
}

int vIsNotEmpty(Vector * v)
{
    return v->properties.actualSize;
}

int vIsEmpty(Vector * v)
{
    return (v->properties.actualSize == 0);
}

inline void vLock(Vector * v)
{

    spin_lock(&(v->properties.lock));

    return;
}

inline void vUnlock(Vector * v)
{
    spin_unlock(&(v->properties.lock));

    return;
}

bool vResize(Vector * v, unsigned long size)
{
    voidp * newData = NULL;
    int i = 0;
    int iMax = 0;
  
    
    if (!v) return false;
    
    if (size == v->properties.size) return true;
    
    vLock(v);
    
    if (size > 0) {
        newData = allocateMemory(sizeof(voidp) * size, "vResize");
        if (!newData) return false;
        
        iMax = (size < v->properties.actualSize ? size : v->properties.actualSize);
        
        for (i = 0; i < iMax; i++) {
            newData[i] = v->data[v->properties.rPos++];
            v->properties.rPos %= v->properties.size;
        }
    }
    
    vClear_NoLock(v);

    kfree((void *) v->data);
    v->data = newData;
    v->properties.wPos = iMax % size;
    v->properties.size = size;
    v->properties.actualSize = iMax;
    
    vUnlock(v);
    
    return true;

}

void vSetIsPointer(Vector * v, bool ip)
{
    v->properties.isPointer = ip;
}

bool vIsPointer(Vector * v)
{
    return v->properties.isPointer;
}

unsigned long vGetActualSize(Vector * v)
{
    return v->properties.actualSize;
}

unsigned long vGetMaximumSize(Vector * v)
{
    return v->properties.size;
}
