/*
 * Copyright 2013 Israel Martín Escalona <imartin@entel.upc.edu>
 * 
 * This file is part of Wipos.
 * 
 * Wipos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Wipos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Wipos.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

/*
 *
 * Functions related with the 2-way TOA algorithm
 */

#include <linux/kernel.h> /* printk */
#include <linux/export.h> /* export symbol */
#include <linux/ieee80211.h> /* mac80211 functions */
#include <linux/skbuff.h> /* mac80211 functions */
#include <linux/sched.h> /* mutex */
#include <linux/timex.h>  /* get_cycles */
#include <linux/errno.h>  
#include <net/mac80211.h>
#include "ieee80211_i.h"
#include "two_way_toa.h"
#include "positioning_utils.h"
#include "queue.h"
#include "vector.h"
#include "k_container.h"

/*
#define RTT_CONTANIER_CREATE qCreate
#define RTT_CONTANIER_RENAME qSetName
#define RTT_CONTANIER_RESIZE qResize
#define RTT_CONTANIER_DELETE qDelete
#define RTT_CONTANIER_CLEAR qClear
#define RTT_CONTANIER_IS_NOT_EMPTY qIsNotEmpty
#define RTT_CONTANIER_IS_EMPTY qIsEmpty
#define RTT_CONTANIER_ADD_DATA qAddData
#define RTT_CONTANIER_GET_NEXT qGetNext
#define RTT_CONTAINER_SIZE qGetActualSize
#define RTT_CONTAINER_MAXIMUM_SIZE qGetMaximumSize
*/

#define RTT_CONTANIER_CREATE vCreate
#define RTT_CONTANIER_RENAME vSetName
#define RTT_CONTANIER_RESIZE vResize
#define RTT_CONTANIER_DELETE vDelete
#define RTT_CONTANIER_CLEAR vClear
#define RTT_CONTANIER_IS_NOT_EMPTY vIsNotEmpty
#define RTT_CONTANIER_IS_EMPTY vIsEmpty
#define RTT_CONTANIER_ADD_DATA vAddData
#define RTT_CONTANIER_GET_NEXT vGetNext
#define RTT_CONTAINER_SIZE vGetActualSize
#define RTT_CONTAINER_MAXIMUM_SIZE vGetMaximumSize

#define RTT_CONTAINER_IS_FULL(queue) (RTT_CONTAINER_SIZE(queue) == RTT_CONTAINER_MAXIMUM_SIZE(queue))

#define TWO_WAY_TOA_STR "TWO_WAY_TOA"
#define DEFAULT_RTT_QUEUE_SIZE 256


/* Global variables */
static bool gather_two_way_toa_state;
static Vector * rtt_queue;
static DECLARE_WAIT_QUEUE_HEAD(rtt_data_readers_blocked);

// Blocking Buffered read variables
static spinlock_t size_to_read_lock;
static unsigned long long size_to_read;
#define SET_SIZE_TO_READ(size)	spin_lock(&size_to_read_lock);size_to_read=size;spin_unlock(&size_to_read_lock);


#ifdef _COMPUTE_TIME_IN_NS_
unsigned long long last_tx_time;
#endif

#ifdef _COMPUTE_TIME_IN_CLOCKS_
unsigned long long last_tx_clock;
#endif


/* GENERAL FUNCTIONS */
void two_way_toa_init(void)
{
    gather_two_way_toa_state = false; 
    
    // Setting up the tx-timestamps container
#ifdef _COMPUTE_TIME_IN_NS_
    last_tx_time = (unsigned long long)-1;
#endif
#ifdef _COMPUTE_TIME_IN_CLOCKS_
    last_tx_clock = (unsigned long long)-1;
#endif
    
    // Setting up the RTT container
    rtt_queue = RTT_CONTANIER_CREATE();
    if (!rtt_queue)
    {
	printk("%s: Two_way_toa_init: Failed to allocate memory for rtt_queue\n", TWO_WAY_TOA_STR);
	return;
    }
    RTT_CONTANIER_RENAME(rtt_queue, TWO_WAY_TOA_STR);
    RTT_CONTANIER_RESIZE(rtt_queue, DEFAULT_RTT_QUEUE_SIZE);
      
    printk("%s: Two-way TOA extensions loaded\n", TWO_WAY_TOA_STR);
}

void two_way_toa_exit(void)
{
    if (rtt_queue)
    {
	RTT_CONTANIER_DELETE(rtt_queue);
	rtt_queue = NULL;
    }
       
    printk("%s: Two-way TOA extensions unloaded\n", TWO_WAY_TOA_STR);
}

void clear_buffer_two_way_toa(void)
{
    RTT_CONTANIER_CLEAR(rtt_queue);
}
EXPORT_SYMBOL(clear_buffer_two_way_toa);

void set_gathering_buffer_two_way_toa(unsigned long samples)
{
    RTT_CONTANIER_RESIZE(rtt_queue, samples);
}
EXPORT_SYMBOL(set_gathering_buffer_two_way_toa);

void enable_gathering_two_way_toa(void)
{
    gather_two_way_toa_state = true;
}
EXPORT_SYMBOL(enable_gathering_two_way_toa);

void disable_gathering_two_way_toa(void)
{
    gather_two_way_toa_state = false;
    RTT_CONTANIER_CLEAR(rtt_queue);
    wake_up_interruptible(&rtt_data_readers_blocked);
}
EXPORT_SYMBOL(disable_gathering_two_way_toa);

unsigned long two_way_toa_data_in_buffer(void)
{
    return RTT_CONTANIER_IS_NOT_EMPTY(rtt_queue);
}
EXPORT_SYMBOL(two_way_toa_data_in_buffer);

inline void two_way_toa_timestamp_tx(struct ieee80211_hdr *hdr, struct wiphy * wiphy)
{
#ifdef _DEBUG_POSITIONING_
    char buffer[32];
#endif

#ifdef _COMPUTE_TIME_IN_NS_
    struct timespec ts;
#endif
    
    // It only works when data gathering is enabled
    if (!gather_two_way_toa_state)
	return;

    if (ieee80211_is_mgmt(hdr->frame_control) || ieee80211_is_ctl(hdr->frame_control))
	return;
    
#ifdef _DEBUG_POSITIONING_
    convert_mac_to_str(wiphy->perm_addr, buffer);
    printk("%s: TX DATA - %s - ", TWO_WAY_TOA_STR, buffer);
#endif
	
#ifdef _COMPUTE_TIME_IN_NS_
    getnstimeofday(&ts);
    last_tx_time = TIMESPEC_TO_NS(ts);
#ifdef _DEBUG_POSITIONING_
    printk(" %llu (seq: %d)\n", last_tx_time, hdr->seq_ctrl);
#endif /* _DEBUG_POSITIONING_ */
#endif /* _COMPUTE_TIME_IN_NS */
	
#ifdef _COMPUTE_TIME_IN_CLOCKS_
    last_tx_clock = get_cycles();
#ifdef _DEBUG_POSITIONING_
    printk(" %llu (seq: %d)\n", (unsigned long long) last_tx_clock, hdr->seq_ctrl);
#endif /* _DEBUG_POSITIONING */
#endif /* _COMPUTE_TIME_IN_CLOCKS_ */
}

inline void two_way_toa_timestamp_rx(TYPE_OF_POSITIONING_DATA last_rx_time, TYPE_OF_POSITIONING_DATA last_rx_clock, struct ieee80211_hw *hw, struct sk_buff *skb)
{
    struct ieee80211_hdr *hdr = NULL;
    TYPE_OF_POSITIONING_DATA newRTT = 0;
#ifdef _DEBUG_POSITIONING_
    char addr[3*ETH_ALEN];
#endif
    
    // It only works when data gathering is enabled
    if (!gather_two_way_toa_state)
	return;
    
    hdr = (struct ieee80211_hdr *)skb->data;
    
#ifdef _COMPUTE_TIME_IN_NS_  
    if (ieee80211_is_ack(hdr->frame_control) && mac_addr_cmp(hdr->addr1, hw->wiphy->perm_addr) && last_tx_time < last_rx_time) 
    {
	newRTT = (TYPE_OF_POSITIONING_DATA)(last_rx_time - last_tx_time);
	RTT_CONTANIER_ADD_DATA(rtt_queue, (voidp) newRTT);
	last_tx_time = (unsigned long long)-1;
#ifdef _DEBUG_POSITIONING_
	convert_mac_to_str(hdr->addr1, addr);
	printk("%s: RX ACK - %s - %llu\n%s: RTT - %llu ns\n", TWO_WAY_TOA_STR, addr, last_rx_time, TWO_WAY_TOA_STR, newRTT);
#endif /* _DEBUG_POSITIONING_ */
#endif /* _COMPUTE_TIME_IN_NS */
	
#ifdef _COMPUTE_TIME_IN_CLOCKS_
    if (ieee80211_is_ack(hdr->frame_control) && mac_addr_cmp(hdr->addr1, hw->wiphy->perm_addr) && last_tx_clock < last_rx_clock) 
    {
	newRTT = (TYPE_OF_POSITIONING_DATA) (last_rx_clock - last_tx_clock);
	RTT_CONTANIER_ADD_DATA(rtt_queue, (voidp) newRTT);
	last_tx_clock = (unsigned long long)-1;
#ifdef _DEBUG_POSITIONING_
	convert_mac_to_str(hdr->addr1, addr);
	printk("%s: RX ACK - %s - %llu\n%s: RTT computed - %llu clocks\n", TWO_WAY_TOA_STR, addr, (TYPE_OF_POSITIONING_DATA) last_rx_clock, TWO_WAY_TOA_STR, newRTT);
#endif /* _DEBUG_POSITIONING_ */
#endif /* _COMPUTE_TIME_IN_CLOCKS_ */
	
	// Wake up only when enough data is available. Reduces the interference with the data gathering.
	if (RTT_CONTAINER_SIZE(rtt_queue) >= size_to_read || RTT_CONTAINER_IS_FULL(rtt_queue))
	    wake_up_interruptible(&rtt_data_readers_blocked);
    }
}

read_return_struct get_next_rtt_two_way_toa(TYPE_OF_POSITIONING_DATA * buffer, unsigned long size)
{ 
    read_return_struct ret;
    container_dato dato;
    unsigned long i = 0;

    i = 0;
    ret.error = dato.error = 0;
    
    SET_SIZE_TO_READ(size);
    while(gather_two_way_toa_state && i < size && dato.error >=0)
    {
	if (wait_event_interruptible(rtt_data_readers_blocked, RTT_CONTAINER_SIZE(rtt_queue) >= size_to_read || RTT_CONTAINER_IS_FULL(rtt_queue)))
	{
	    ret.error = -ERESTARTSYS;
	    break;   
	}
	
	dato = RTT_CONTANIER_GET_NEXT(rtt_queue);
	while(i < size && dato.error >= 0)
	{
	    buffer[i++] = (TYPE_OF_POSITIONING_DATA) dato.item;	    
	    dato = RTT_CONTANIER_GET_NEXT(rtt_queue);
#ifdef _DEBUG_POSITIONING_
	    printk("%s: New RTT enqueued for reading --> %llu clocks with %d error\n", TWO_WAY_TOA_STR, (unsigned long long)dato.item, dato.error);
#endif
	}
    }
    SET_SIZE_TO_READ(size_to_read - i);
    
    ret.data_read = i;
    
    if (!gather_two_way_toa_state) 
    {
	ret.error = -ENODATA;
    }

    return ret;
}
EXPORT_SYMBOL(get_next_rtt_two_way_toa);

bool is_two_way_toa_enabled(void)
{
    return gather_two_way_toa_state;
}

