/*
 * Copyright 2013 Israel Martín Escalona <imartin@entel.upc.edu>
 * 
 * This file is part of Wipos.
 * 
 * Wipos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Wipos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Wipos.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <linux/slab.h>
#include <linux/limits.h>
#include "queue.h"

Queue * qCreate(void)
{
    Queue * q = NULL;

    q = (Queue *) kmalloc(sizeof(Queue), GFP_ATOMIC);
    if (! q)
    {
	printk(KERN_ALERT "qCreate:kmalloc failed\n");
	return q;
    }

    INIT_LIST_HEAD(&(q->list));

    spin_lock_init(&(q->lock));
    
    q->size = 0;
    q->max_size = ULONG_MAX;
    q->isPointer = false;
    
    qSetName(q, "No name");

    return q;
}

void qSetName(Queue *q, char * name)
{
    
    int i = 0;
    
    qLock(q);
    
    while (i < MAX_QNAME_SIZE && *name != '\0')
    {
	q->name[i] = *name;
	name++;
    }
    
    if (i == MAX_QNAME_SIZE && *name != '\0')
    {
	printk(KERN_ALERT "qSetName: name larger than %d characters\n", MAX_QNAME_SIZE-1);
    }
    
    qUnlock(q);
    
    return;

}

// Queue must be empty!
void qDelete(Queue * q)
{
    qClear(q);
    kfree(q);

    return;
}

void qClear(Queue * q)
{
    QueueItem * qi = NULL;
    
    qLock(q);

    if (q->isPointer)
    {
	while (!list_empty(&(q->list)))
	{
	    qi = list_first_entry(&(q->list), QueueItem, list);
	    list_del(&(qi->list));
	    if (qi->item) kfree((voidp) qi->item); // added to delete each item 
	    kfree(qi);
	}
    }
    else
    {
	while (!list_empty(&(q->list)))
	{
	    qi = list_first_entry(&(q->list), QueueItem, list);
	    list_del(&(qi->list));
	    kfree(qi);
	}
    }
    
    q->size = 0;
    qUnlock(q);
}

void qSetPointer(Queue * q, bool isPointer)
{
    q->isPointer = isPointer;
}

bool qIsPointer(Queue * q)
{
    return q->isPointer;
}

void qAddData(Queue * q, voidp item)
{
    QueueItem * qi = NULL;

    qi = (QueueItem *) kmalloc(sizeof(QueueItem), GFP_ATOMIC);
    if (! qi)
    {
	printk(KERN_ALERT "qEnqueue:kmalloc failed\n");
	return;
    }
    
    // Fixing queue size
    if (q->size < q->max_size)
    {
	// Adding more elements is still possible
	q->size++;
	
	// Adding the new element
	qi->item = item;

	qLock(q);
	list_add_tail(&(qi->list), &(q->list));
	qUnlock(q);
    } 
    else
    {
	kfree(qi);
	printk(KERN_ALERT "qEnqueue: Queue %s full\n", q->name);
    }

    return;
}

container_dato qGetNext(Queue * q)
{
    QueueItem *   qi    = NULL;
    container_dato dato;

    qLock(q);
    
    if (list_empty(&(q->list)))
    {
	dato.item = NULL;
	dato.error = -ERR_CONTAINER_IS_EMPTY;
	return dato;
    }
    else
	dato.error = 0;

    qi = list_first_entry(&(q->list), QueueItem, list);

    dato.item = qi->item;

    list_del(&(qi->list));

    kfree(qi);
    
    // Fixing queue size
    q->size--;

    qUnlock(q);

    return dato;
}

int qIsEmpty(Queue * q)
{
    int ret = 0;

    qLock(q);
    ret = list_empty(&(q->list));
    qUnlock(q);

    return ret;
}

inline
int qIsNotEmpty(Queue * q)
{
    return !qIsEmpty(q);
}

int qLock(Queue * q)
{

    spin_lock(&(q->lock));

    return 0;
}

void qUnlock(Queue * q)
{

    spin_unlock(&(q->lock));

    return;
}

void qResize(Queue * q, unsigned long new_max_size)
{

    container_dato	dato;
    int			i   = 0;
    
    if (qIsEmpty(q))
    {
	return;
    }
    
    if (new_max_size < q->size)
    {
	// Dequeueing the exceeding size
	for (i = 0; i < new_max_size - q->size; i++)
	{
	    dato = qGetNext(q);
	    kfree(dato.item);
	}
	q->size = new_max_size;
	
	printk(KERN_ALERT "qSetMaxSize: Fixing the maximum size for the queue %s. Few elements were dropped\n", q->name);
    }
    
    q->max_size = new_max_size;
    
    return;

}

volatile unsigned long qGetActualSize(Queue * q)
{
    return q->size;
}

volatile unsigned long qGetMaximumSize(Queue * q)
{
    return q->max_size;
}
