/*
 * Copyright 2013 Israel Martín Escalona <imartin@entel.upc.edu>
 * 
 * This file is part of Wipos.
 * 
 * Wipos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Wipos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Wipos.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

/*
 * Functions related with the 2-way TOA algorithm
 */

#include <linux/kernel.h> /* printk */
#include <linux/export.h> /* export symbol */
#include <linux/ieee80211.h> /* mac80211 functions */
#include <linux/skbuff.h> /* mac80211 functions */
#include <linux/sched.h> /* mutex */
#include <linux/timex.h>  /* get_cycles */
#include <linux/errno.h>  
#include <net/mac80211.h>
#include "ieee80211_i.h"
#include "passive_tdoa_common.h"
#include "positioning_utils.h"

/*
#include "queue.h" 
#define TDOA_CONTANIER_CREATE qCreate
#define TDOA_CONTANIER_RENAME qSetName
#define TDOA_CONTANIER_RESIZE qResize
#define TDOA_CONTANIER_DELETE qDelete
#define TDOA_CONTANIER_CLEAR qClear
#define TDOA_CONTANIER_IS_NOT_EMPTY qIsNotEmpty
#define TDOA_CONTANIER_IS_EMPTY qIsEmpty
#define TDOA_CONTANIER_ADD_DATA qAddData
#define TDOA_CONTANIER_GET_NEXT qGetNext
#define TDOA_CONTAINER_SIZE qGetActualSize
#define TDOA_CONTAINER_MAXIMUM_SIZE qGetMaximumSize
*/

#include "vector.h"
#define TDOA_CONTANIER_CREATE vCreate
#define TDOA_CONTANIER_RENAME vSetName
#define TDOA_CONTANIER_RESIZE vResize
#define TDOA_CONTANIER_DELETE vDelete
#define TDOA_CONTANIER_CLEAR vClear
#define TDOA_CONTANIER_IS_NOT_EMPTY vIsNotEmpty
#define TDOA_CONTANIER_IS_EMPTY vIsEmpty
#define TDOA_CONTANIER_ADD_DATA vAddData
#define TDOA_CONTANIER_GET_NEXT vGetNext
#define TDOA_CONTAINER_SIZE vGetActualSize
#define TDOA_CONTAINER_MAXIMUM_SIZE vGetMaximumSize

#define TDOA_CONTAINER_IS_FULL(queue) (TDOA_CONTAINER_SIZE(queue) == TDOA_CONTAINER_MAXIMUM_SIZE(queue))

#define DEFAULT_TDOA_QUEUE_SIZE 256
#define PASSIVE_TDOA_STR "PASSIVE TDOA (COMMON)"
#define TX_PACKET_RECEIVED (0)
#define RX_PACKET_RECEIVED (1)

/* Global variables */
static bool gather_passive_tdoa_state;	// Flag for gathering data or not
//static Queue * tdoa_queue;
static Vector * tdoa_queue;	// TDOA data
static u8 activeNodeAddr[ETH_ALEN];	// Address of the active node
static char activeNodeAddrStr[3*ETH_ALEN];	// String of the address of the active node
static DECLARE_WAIT_QUEUE_HEAD(passive_tdoa_data_readers_blocked);	// Queue for the blocking read
static int tx_status;

static spinlock_t size_to_read_lock;
static unsigned long long size_to_read;
#define SET_SIZE_TO_READ(size)	spin_lock(&size_to_read_lock);size_to_read=size;spin_unlock(&size_to_read_lock);

#ifdef _COMPUTE_TIME_IN_NS_
static unsigned long long last_tx_time;
#endif

#ifdef _COMPUTE_TIME_IN_CLOCKS_
static cycles_t last_tx_clock;
#endif

/* PROTOTYPES */
void setSizeToRead(unsigned long long size);


/* GENERAL FUNCTIONS */
void common_passive_tdoa_init(void)
{
    // TODO: remove after debug stage is over
    //setActiveNodeAddressByStr("00:02:e3:45:ff:b5"); 
    //setActiveNodeAddressByStr("00:21:6A:9D:E6:12");
    // ---------------------------------------
    gather_passive_tdoa_state = false; // TODO: Turn into false when debug stage is over 
    tdoa_queue = TDOA_CONTANIER_CREATE();
    if (!tdoa_queue)
    {
	printk("%s: <<common_passive_tdoa_init>> Failed to allocate memory for tdoa_queue\n", PASSIVE_TDOA_STR);
	return;
    }
    TDOA_CONTANIER_RENAME(tdoa_queue, PASSIVE_TDOA_STR);
    TDOA_CONTANIER_RESIZE(tdoa_queue, DEFAULT_TDOA_QUEUE_SIZE);
    
    tx_status = RX_PACKET_RECEIVED;
    
    printk("%s: Common passive TDOA extensions loaded\n", PASSIVE_TDOA_STR);
}

void common_passive_tdoa_exit(void)
{
    if (tdoa_queue)
    {
	TDOA_CONTANIER_DELETE(tdoa_queue);
	tdoa_queue = NULL;
    }
    printk("%s: Common passive TDOA extensions unloaded\n", PASSIVE_TDOA_STR);
}

void clear_buffer_common_passive_tdoa(void)
{
    TDOA_CONTANIER_CLEAR(tdoa_queue);
}
EXPORT_SYMBOL(clear_buffer_common_passive_tdoa);

void set_gathering_buffer_common_passive_tdoa(unsigned long samples)
{
    TDOA_CONTANIER_RESIZE(tdoa_queue, samples);
}
EXPORT_SYMBOL(set_gathering_buffer_common_passive_tdoa);

void enable_gathering_common_passive_tdoa(void)
{
    gather_passive_tdoa_state = true;
}
EXPORT_SYMBOL(enable_gathering_common_passive_tdoa);

void disable_gathering_common_passive_tdoa(void)
{
    gather_passive_tdoa_state = false;
    TDOA_CONTANIER_CLEAR(tdoa_queue);
    wake_up_interruptible(&passive_tdoa_data_readers_blocked);
}
EXPORT_SYMBOL(disable_gathering_common_passive_tdoa);

unsigned long common_passive_tdoa_data_in_buffer(void)
{   
    return TDOA_CONTANIER_IS_NOT_EMPTY(tdoa_queue);
}
EXPORT_SYMBOL(common_passive_tdoa_data_in_buffer);

inline void common_passive_tdoa_timestamp_rx_data(TYPE_OF_POSITIONING_DATA last_rx_time, TYPE_OF_POSITIONING_DATA last_rx_clock, struct sk_buff *skb)
{
    struct ieee80211_hdr *hdr = NULL;
    char buffer[18];
    
    
    // It only works when data gathering is enabled
    if (!gather_passive_tdoa_state)
	return;
    
    hdr = (struct ieee80211_hdr *)skb->data;

    if (hdr && ieee80211_has_tods(hdr->frame_control) && mac_addr_cmp(activeNodeAddr, hdr->addr2))
    {
#ifdef _DEBUG_POSITIONING_	
	// TODO: Fill this
#endif /* _DEBUG_POSITIONING_ */
	
#ifdef _COMPUTE_TIME_IN_NS_  
	last_tx_time = last_rx_time;
#ifdef _DEBUG_POSITIONING_
	convert_mac_to_str(hdr->addr1, buffer);
	printk("%s: RX DATA FROM %s TO AP %s AT %llu ns\n", PASSIVE_TDOA_STR, activeNodeAddrStr, buffer, last_tx_time);
#endif /* _DEBUG_POSITIONING_ */
#endif /* _COMPUTE_TIME_IN_NS */
	
#ifdef _COMPUTE_TIME_IN_CLOCKS_
	last_tx_clock = last_rx_clock;
#ifdef _DEBUG_POSITIONING_
	printk("%s: RX DATA FROM %s TO AP %s AT %llu clocks\n", PASSIVE_TDOA_STR, activeNodeAddrStr, buffer, last_tx_clock);
#endif /* _DEBUG_POSITIONING_ */
#endif /* _COMPUTE_TIME_IN_CLOCKS_ */
	
	tx_status = TX_PACKET_RECEIVED;
    }
}

inline void common_passive_tdoa_timestamp_rx_ack(TYPE_OF_POSITIONING_DATA last_rx_time, TYPE_OF_POSITIONING_DATA last_rx_clock, struct ieee80211_hw *hw, struct sk_buff *skb)
{
    struct ieee80211_hdr *hdr = NULL;
    TYPE_OF_POSITIONING_DATA newTDOA = 0;
    char buffer[18];

    
    // It only works when data gathering is enabled
    if (!gather_passive_tdoa_state)
	return;
      
    hdr = (struct ieee80211_hdr *)skb->data;
    
    if (tx_status == TX_PACKET_RECEIVED && hdr && ieee80211_is_ack(hdr->frame_control) && mac_addr_cmp(activeNodeAddr, hdr->addr1)) 
    {
#ifdef _DEBUG_POSITIONING_
	convert_mac_to_str(hdr->addr2, buffer);
#endif
	
#ifdef _COMPUTE_TIME_IN_NS_  
	newTDOA = last_rx_time - last_tx_time;
	TDOA_CONTANIER_ADD_DATA(tdoa_queue, (voidp) newTDOA);
#ifdef _DEBUG_POSITIONING_
	printk("%s: RX ACK FROM AP %s TO %s AT %llu ns\n%s: TDOA - %llu ns\n", PASSIVE_TDOA_STR, buffer, activeNodeAddrStr, last_rx_time, PASSIVE_TDOA_STR, newTDOA);
#endif /* _DEBUG_POSITIONING_ */
#endif /* _COMPUTE_TIME_IN_NS */
	
#ifdef _COMPUTE_TIME_IN_CLOCKS_
	newTDOA = (TYPE_OF_POSITIONING_DATA) (last_rx_clock - last_tx_clock);
	TDOA_CONTANIER_ADD_DATA(tdoa_queue, (voidp) newTDOA);
#ifdef _DEBUG_POSITIONING_
	printk("%s: RX ACK FROM AP %s TO %s AT %llu clocks\n%s: TDOA - %llu clocks\n", PASSIVE_TDOA_STR, buffer, activeNodeAddrStr, (TYPE_OF_POSITIONING_DATA) last_rx_clock, PASSIVE_TDOA_STR, newTDOA);
#endif /* _DEBUG_POSITIONING_ */
#endif /* _COMPUTE_TIME_IN_CLOCKS_ */
	
	tx_status = RX_PACKET_RECEIVED;
	
	// Wake up only when enough data is available. Reduces the interference with the data gathering.
	if (TDOA_CONTAINER_SIZE(tdoa_queue) >= size_to_read || TDOA_CONTAINER_IS_FULL(tdoa_queue))
	    wake_up_interruptible(&passive_tdoa_data_readers_blocked);
    }
}

read_return_struct get_next_passive_tdoa_measurement(TYPE_OF_POSITIONING_DATA * buffer, unsigned long size)
{ 
    read_return_struct ret;
    container_dato dato;
    unsigned long i = 0;

    i = 0;
    ret.error = dato.error = 0;
    
    SET_SIZE_TO_READ(size);
    while(gather_passive_tdoa_state && i < size && dato.error >=0)
    {
	if (wait_event_interruptible(passive_tdoa_data_readers_blocked, TDOA_CONTAINER_SIZE(tdoa_queue) >= size_to_read || TDOA_CONTAINER_IS_FULL(tdoa_queue)))
	{
	    ret.error = -ERESTARTSYS;
	    break;
	}
	
	dato = TDOA_CONTANIER_GET_NEXT(tdoa_queue);
	while(i < size && dato.error >= 0)
	{
	    buffer[i++] = (TYPE_OF_POSITIONING_DATA) dato.item;
	    dato = TDOA_CONTANIER_GET_NEXT(tdoa_queue);
	}
    }
    
    SET_SIZE_TO_READ(size_to_read - i);
    ret.data_read = i;
    
    if (!gather_passive_tdoa_state) 
    {
	ret.error = -ENODATA;
    }
    
    return ret;
}
EXPORT_SYMBOL(get_next_passive_tdoa_measurement);

bool is_common_passive_tdoa_enabled(void)
{
    return gather_passive_tdoa_state;
}

void setActiveNodeAddress(u8 * addr)
{
    int i = 0;
    
    for (i = 0; i < ETH_ALEN; i++)
    {
	activeNodeAddr[i] = addr[i];
    }
    
    convert_mac_to_str(activeNodeAddr, activeNodeAddrStr);
}

void setActiveNodeAddressByStr(char * str)
{ 
    int i = 0;
    
    // Generate the mac address of the active node
    convert_str_to_mac(str, activeNodeAddr);
    
    // Copying the string
    i = 0;
    while (str[i] != '\0' && i < 3 * ETH_ALEN)
    {
	activeNodeAddrStr[i] = str[i];
	i++;
    }
    
    if (str[i] != '\0')
	activeNodeAddrStr[3*ETH_ALEN-1] = '\0';
#ifdef _DEBUG_POSITIONING_
    printk("%s: Listening TDOAs for the active node %s\n", PASSIVE_TDOA_STR, activeNodeAddrStr);
#endif
    
}
EXPORT_SYMBOL(setActiveNodeAddressByStr);

const u8 * getActiveNodeAddress(void)
{
    return activeNodeAddr;
}



